// yolooo
const cols = 5;
const rows = 14;

// options
const config = {
  transitionSpeed: 0.05, // Transitions speed
  maxWeight: 2, // Maximum weight for hovered col
  minWeight: 0.8, // Minimum weight for farest col
  backgroundColor: 00       
};

let images = [];
for (let i = 1; i <= cols * rows; i++) {
  let num = i.toString().padStart(4, '0');
  images.push(`https://ateliers.esad-pyrenees.fr/web/archives/2024-2025/Ruisseau/baudreix_videos/frames/${num}.gif`);
}

let imgObjects = [];
let loaded = 0;
const imgRatio = 400 / 225;

let currentWeights = Array(cols).fill(1);  // initial equal widths
let targetWeights = Array(cols).fill(1);

function preload() {
  for (let i = 0; i < images.length; i++) {
    imgObjects[i] = loadImage(images[i], 
      () => { loaded++; },
      () => { console.log(`Failed to load image ${i}`); }
    );
  }
}

function setup() {
  createCanvas(windowWidth, windowHeight);
  normalizeWeights(targetWeights);
}


function draw() {

  background(config.backgroundColor);
  
  // Update target weights according mouse pos
  updateTargetWeights();
  
  // Animate column widths
  animateColumnWeights();
  
  // Draw the image grid
  drawImageGrid();
  
}

function drawImageGrid() {
  const colWidths = getColsWidths();
  let xOffset = 0;
  let imgIndex = 0;

  // Apply scroll offset
  let yOffset = 0;

  for (let col = 0; col < cols; col++) {
    const colW = colWidths[col];
    const imgH = colW / imgRatio;

    for (let i = 0; i < rows; i++) {
      const y = yOffset + i * imgH;
      
      // try to be smart, only draw images that are visible on screen (with 2 × img height buffer)
      if (y > -imgH * 2 && y < height + imgH) {
        const img = imgObjects[imgIndex % imgObjects.length];
        if (img) {
          image(img, xOffset, y, colW, imgH);
        }
      }
      imgIndex++;
    }

    // dividers
    stroke(255, 50);
    line(xOffset, 0, xOffset, height);
    
    xOffset += colW;
  }
}

function updateTargetWeights() {
  let mouseCol = floor(map(mouseX, 0, width, 0, cols));
  mouseCol = constrain(mouseCol, 0, cols - 1);

  for (let i = 0; i < cols; i++) {
    let dist = abs(i - mouseCol);
    targetWeights[i] = lerp(config.maxWeight, config.minWeight, dist / (cols - 1));
  }

  // Normalize so that total weight = cols
  normalizeWeights(targetWeights);
}

function animateColumnWeights() {
  for (let i = 0; i < cols; i++) {
    // slurp transition with config speed
    currentWeights[i] = lerp(currentWeights[i], targetWeights[i], config.transitionSpeed);
  }
  normalizeWeights(currentWeights);
}

function getColsWidths() {
  const totalWeight = currentWeights.reduce((sum, w) => sum + w, 0);
  return currentWeights.map(w => (w / totalWeight) * width);
}

function normalizeWeights(weights) {
  // Calculates the current sum of all weights 
  const sum = weights.reduce((a, b) => a + b, 0);
  // Adjusts each weight proportionally so that their total sum equals the number of cols 
  for (let i = 0; i < weights.length; i++) {
    weights[i] = (weights[i] / sum) * cols;
  }
}


function windowResized() {
  resizeCanvas(windowWidth, windowHeight);
}